###############################################################################
### ZineBomb
###

use strict;

use Gamma::XML;

###############################################################################
### package Gamma::Functions::Templates
###

package Gamma::Functions::Templates;

use File::Basename;
use LWP::UserAgent;
use HTTP::Request;

sub fetch_channels_and_categories {
    my ($self) = @_;
    my $channels = {};
    my $categories = {};
    my $source_id = 1;
    for my $source (@$Gamma::RSS_SOURCES) {
        my $path = dirname($source->{path});
        my $current_channels = $channels->{$source_id} = {};
        my $current_categories = $categories->{$source_id} = {
            0 => {
                id => "$source_id,0",
                source_id => $source_id,
                order_id => $source_id*10000,
                category_id => 0,
                root => $path,
                title => 'default',
                description => '',
                channels => [],
                rotate => 0,
                new_rotate => 0,
            },
        };
        if (open(CATEGORIES, "<$path/data/user-categories.dat")) {
            while (my $line = <CATEGORIES>) {
                chomp($line);
                my ($category_id, $title, $description, $template, $expire) =
                        split(/\|/, $line);
                $current_categories->{$category_id} = {
                    id => "$source_id,$category_id",
                    source_id => $source_id,
                    category_id => $category_id,
                    order_id => $source_id*10000+$category_id*100,
                    root => $path,
                    title => $title,
                    description => $description,
                    channels => [],
                    rotate => 0,
                    new_rotate => 0,
                };
            }
            close(CATEGORIES);
        }
        open(CHANNELS, "<$path/data/user-channels.dat")
            or $self->fail("cannot read file", "$path/data/usr-channels.dat", $!);
        while (my $line = <CHANNELS>) {
            chomp($line);
            my ($channel_id, $title, $description, $rss_url, $html_url, $category_id)
                    = split(/\|/, $line);
            $category_id = 0 unless $category_id;
            my $channel = $current_channels->{$channel_id} = {
                id => "$source_id,$channel_id",
                source_id => $source_id,
                channel_id => $channel_id,
                category_id => $category_id,
                order_id => $source_id*10000+$category_id*100+$channel_id,
                root => $path,
                path => "$path/data/cache/user-$channel_id.xml",
                title => $title,
                description => $description,
                xml_url => $rss_url,
                html_url => $html_url,
                rotate => 0,
                new_rotate => 0,
            };
            push @{$current_categories->{$category_id}->{channels}}, $channel;
        }
        close(CHANNELS);
        if (open(ROTATE, "<$path/data/user-rotate.dat")) {
            while (my $line = <ROTATE>) {
                chomp($line);
                my ($type,$id,$value) = split(/\|/, $line);
                if ($type eq 'channel') {
                    $current_channels->{$id}->{rotate} = $value;
                    $current_channels->{$id}->{new_rotate} = $value;
                }
                if ($type eq 'category') {
                    $current_categories->{$id}->{rotate} = $value;
                    $current_categories->{$id}->{new_rotate} = $value;
                }
            }
            close(ROTATE);
        }
        $source_id ++;
    }
    return ($channels, $categories);
}

sub update_channels_and_categories {
    my ($self) = @_;
    return unless $self->{channels_cache} and $self->{categories_cache};
    my $channels = $self->{channels_cache};
    my $categories = $self->{categories_cache};
    for my $source_id (keys %$categories) {
        my $path = $categories->{$source_id}->{0}->{root};
        my $rotates = [];
        for my $category (values %{$categories->{$source_id}}) {
            push @$rotates,
                ["category", $category->{category_id}, $category->{new_rotate}]
                    if $category->{new_rotate};
        }
        for my $channel (values %{$channels->{$source_id}}) {
            push @$rotates,
                ["channel", $channel->{channel_id}, $channel->{new_rotate}]
                    if $channel->{new_rotate};
        }
        if (@$rotates) {
            open(ROTATE, ">$path/data/user-rotate.dat")
                or return $self->fail("cannot write to file",
                        "$path/data/user-rotate.dat", $!);
            for my $line (@$rotates) {
                my ($type, $id, $value) = @$line;
                print ROTATE "$type|$id|$value\n";
            }
            close(ROTATE);
        }
    }
}

sub find_channels {
    my ($self) = @_;
    my $channels = [];
    my $source_id = 1;
    for my $source (@$Gamma::RSS_SOURCES) {
        my $path = dirname($source->{path});
        open (CHANNELS, "<$path/data/user-channels.dat")
            or $self->fail("cannot read file", "$path/data/user-channels.dat", $!);
        while (my $line = <CHANNELS>) {
            chomp($line);
            my ($channel_id, $title, $description, $xml_url, $html_url) = split(/\|/, $line);
            push @$channels, {
                id => "$source_id,$channel_id",
                source_id => $source_id,
                channel_id => $channel_id,
                order_id => $source_id*10000+$channel_id,
                root => $path,
                path => "$path/data/cache/user-$channel_id.xml",
                title => $title,
                description => $description,
                xml_url => $xml_url,
                html_url => $html_url,
            };
            @$channels = sort { $a->{order_id} <=> $b->{order_id} } @$channels;
        }
        close (CHANNELS);
        $source_id ++;
    }
    return $channels;
}

sub trigger_channel {
    my ($self, $channel) = @_;
    my $config_path = $channel->{root} . "/data/config.pl";
    my $ssi_url = "";
    do {
        local $SIG{__DIE__};
        eval {
            no strict;
            require $config_path;
            $ssi_url = $FEEDSEARCH_SSI_URL;
        };
        return $self->fail("cannot read channel configuration", $config_path, $@)
            if $@;
        return $self->fail("cannot read value of FEEDSEARCH_SSI_URL", $config_path, $@)
            unless $ssi_url;
    };
    $ssi_url .= "/" unless $ssi_url =~ /\/$/;
    $ssi_url .= 'feedsearch_ssi.cgi';
    my $url = "$ssi_url?channel=user-$channel->{channel_id}.xml";
    my $ua = new LWP::UserAgent;
    my $request = new HTTP::Request(GET => $url);
    my $response = $ua->request($request);
    return $self->fail("cannot trigger channel", $url, $response->status_line)
        unless $response->is_success;
    return 1;
}

sub make_channel {
    my ($self, $key, $id) = @_;
    unless ($self->{rss_cache}) {
        $self->{rss_cache} = {};
        my ($channels, $categories) = $self->fetch_channels_and_categories;
        $self->{channels_cache} = $channels;
        $self->{categories_cache} = $categories;
        for my $source_id (keys %$categories) {
            for my $channel (values %{$channels->{$source_id}}) {
                $self->{rss_cache}->{$channel->{id}} = $channel;
            }
            for my $category (values %{$categories->{$source_id}}) {
                $self->{rss_cache}->{'cat-'.$category->{id}} = $category;
            }
        }
    }
    return $self->fail("unknown rss channel or category", "{$key:$id}")
            unless (exists $self->{rss_cache}->{$id});
    if ($id =~ /^cat-/) {
        my $category = $self->{rss_cache}->{$id};
        $self->parse_category($category);
        return $category;
    }
    else {
        my $channel = $self->{rss_cache}->{$id};
        $self->parse_channel($channel);
        return $channel;
    }
}

sub parse_category {
    my ($self, $category) = @_;
    return if $category->{data};
    my $data = [];
    for my $channel (@{$category->{channels}}) {
        $self->parse_channel($channel);
        push @$data, @{$channel->{data}};
    }
    $category->{data} = $data;
}

sub parse_channel {
    my ($self, $channel) = @_;
    return if $channel->{data};
    return unless $self->trigger_channel($channel);
    my $path = $channel->{path};
    if (open(CHANNELS, "<$path")) {
        my @lines = <CHANNELS>;
        shift @lines;
        my $content = join('', @lines);
        close(CHANNELS);
        return $self->fail("no rss data", $path)
                unless ($content);
        my @path = ();
        my $chars = "";
        my ($title, $description, $link);
        my $data = [];
        my $parser = new Gamma::XML(undef, $path,
            sub {
                my ($none, $expat, $element, %attributes) = @_;
                push @path, $element;
                $chars = "";
            },
            sub {
                my ($none, $expat, $element) = @_;
                my $path = join('/', @path);
                $link = $chars if $path eq 'rss/channel/item/link';
                $title = $chars if $path eq 'rss/channel/item/title';
                $description = $chars if $path eq 'rss/channel/item/description';
                s/[\x80-\xFF]/ /g for ($link, $title, $description);
                my $domain = $link;
                $domain =~ s{^[^/]+//([^/]+).*$}{$1};
                push @$data, {
                    link => $link,
                    domain => $domain,
                    title => $title,
                    description => $description,
                } if ($path eq 'rss/channel/item');
                pop @path;
            },
            sub {
                my ($none, $expat, $string) = @_;
                $chars .= $string;
            },
        );
        $parser->parse($content);
        $channel->{data} = $data;
    }
    else {
        return $self->fail("cannot read file", $path, $!);
    }
}

sub prepare_templates {
    my ($self, $templates, $sending, $list, $user) = @_;
    my $index_entry = $list->{archive_template};
    my $from_line = $templates->{template_from};
    my $to_line = $templates->{template_to};
    my $subject_line = $templates->{template_subject};
    my $text_body = $templates->{template_text};
    my $html_body = $templates->{template_html};
    my $archive_body = $templates->{template_archive};
    for my $template ($index_entry, $from_line, $to_line, $subject_line,
                $text_body, $html_body, $archive_body) {
        $template =~ s/\{(\w+)\}/$self->substitute_template_variable($sending, $list, $user, $1)/ge;
    }
    for my $template ($from_line, $to_line, $subject_line, $text_body) {
        $template =~ s/\{(\w+):(\d+,\d+)(?:,?(\w+)?=(\d+))?\}/$self->substitute_rss_variable($list, $user, 'text', $1, $2, $3, $4)/ge;
    }
    $html_body =~ s/\{(\w+):(\d+,\d+)(?:,?(\w+)?=(\d+))?\}/$self->substitute_rss_variable($list, $user, 'html', $1, $2, $3, $4)/ge;
    $archive_body =~ s/\{(\w+):(\d+,\d+)(?:,?(\w+)?=(\d+))?\}/$self->substitute_rss_variable($list, $user, 'archive', $1, $2, $3, $4)/ge;
    $index_entry =~ s/\{(\w+):(\d+,\d+)(?:,?(\w+)?=(\d+))?\}/$self->substitute_rss_variable($list, $user, 'archive', $1, $2, $3, $4)/ge;
    $self->update_channels_and_categories;
    for ($text_body) {
        s/<[^>]*>/ /g;
        s/&lt;/</g;
        s/&gt;/>/g;
        s/&quot;/"/g;
        s/&nbsp;/ /g;
        s/&amp;/&/g;
    }
    return {
        index_entry => $index_entry,
        from => $from_line,
        to => $to_line,
        subject => $subject_line,
        text_body => $text_body,
        html_body => $html_body,
        archive_body => $archive_body,
    };
}

sub substitute_template_variable {
    my ($self, $sending, $list, $user, $key) = @_;
    return $list->{$key} if exists $list->{$key};
    return $user->{$key} if exists $user->{$key};
    return ${$Gamma::{uc $key}} if exists $Gamma::{uc $key};
    return "$Gamma::SCRIPT_URL/do_unsubscribe?code=$user->{code}\&list_id=$list->{id}"
        if $key eq 'unsubscribe_url';
    return "$Gamma::SCRIPT_URL/show_manage?code=$user->{code}"
        if $key eq 'manage_url';
    return "$Gamma::SCRIPT_URL/open?id=$sending->{id},$user->{id}"
        if $key eq 'open_url';
    return sprintf($Gamma::ARCHIVE_MESSAGE, $sending->{stamp})
        if $key eq 'message_url';
    my ($year, $month, $day, $hour, $minute, $second) =
        unpack("a4a2a2a2a2a2", $sending->{stamp});
    return "$month-$day-$year" if $key eq 'date';
    return "$hour:$minute" if $key eq 'time';
    $self->fail("unknown template variable", "{$key}");
    return uc "{$key}";
}

sub substitute_rss_variable {
    my ($self, $list, $user, $type, $key, $id, $mode, $count) = @_;
    if ($key eq 'cat') {
        $id = "cat-$id";
        $key = 'rss';
    }
    my $channel = $self->make_channel($key, $id) or return uc "{$key:$id}";
    return uc "{$key:$id}" unless $channel->{data};
    return $channel->{title} if $key eq 'title';
    return $channel->{description} if $key eq 'description';
    return $channel->{xml_url} if $key eq 'xml';
    return $channel->{html_url} if $key eq 'html';
    if ($key eq 'rss') {
        my @items = @{$channel->{data}};
        my $offset = 0;
        if ($mode eq 'rotate') {
            $offset = $channel->{rotate};
            $offset = 0 if $offset >= scalar(@items);
            $channel->{new_rotate} = $offset+$count;
        }
        @items = @items[$offset .. scalar(@items)-1];
        @items = @items[0 .. $count-1] if $count and $count < scalar(@items);
        
        my $template =
                $type eq 'text' ? 'rss_text.txt' :
                $type eq 'html' ? 'rss_html.html' : 'rss_archive.html';
        return $self->template("rss/$template",
            channel => $channel, items => \@items);
    }
    $self->fail("unknown rss variable", "{$key:$id}");
    return uc "{$key}";
}

1;

###############################################################################
###############################################################################
